<?php
require_once 'abstract.php';
/**
 * Mage_Shell_MCCustomerImporter Class
 * Multichem's customer import script:
 *
 *   1. Get CSV from remote server using SFTP
 *   2. Parse CSV for correctness
 *      2.a Valid record: add to processing list
 *      2.b Invalid record: add to error report
 *   3. Loop through rows, checking for an existing record
 *      3.a New record: create record
 *      3.b Existing record: update record
 *   4. Log failed insert/update operations
 *   5. Report on invalid, successful and unsuccessful data
 *
 * @author murrow
 *
 *   Note: for customer codes as array keys, we use a prefix 'c' so
 *   we don't have to worry about integer array keys being reset by
 *   array_shift, etc.
 */


class Mage_Shell_MCCustomerImporter extends Mage_Shell_Abstract {
	
	private $root_directory = 'mc_customer_import';
	private $csv_basename   = 'customers.csv';
	private $csv_filepath   = FALSE;
	private $error_log      = FALSE;
	private $success_log    = FALSE;
	private $row_count      = 27;
	private $debugging      = FALSE;
	private $all_rows       = array();
	private $good_rows      = array();
	private $incorrect_rows = array();
	private $bad_rows       = array();
	private $errors         = array();
	
	public function _construct() {
		
		//      parent::__construct();
		$this->configuration = Mage::getConfig()->getNode('global/resources/mc_ssh');
		
		$this->root_directory =  Mage::getBaseDir('var') . DIRECTORY_SEPARATOR . (string)$this->configuration->local_directory; // $this->_getRootPath() .
		$this->error_log =  $this->root_directory. DIRECTORY_SEPARATOR .  'error.'.date('YmdH').'.log';
		$this->success_log =  $this->root_directory. DIRECTORY_SEPARATOR .  'success.'.date('YmdH').'.log';
		$this->csv_basename   = (string)$this->configuration->local_csv_file_basename;
		$this->csv_filepath   = $this->root_directory . DIRECTORY_SEPARATOR . $this->csv_basename;
		$this->checkRootDirectory();
		$this->db = Mage::getSingleton('core/resource')->getConnection('core_read');
		
		return $this;
	}
	
	public function run() {
		
		if (isset($this->_args['debug'])) {
			$this->debugging = TRUE;
		}
		
		if (isset($this->_args['help'])) {
			echo $this->usageHelp();
		} else {
			if($this->getFile() === TRUE) {
				if($this->parseFile() === TRUE) {
					$this->log('Processing', 'Completed processing');
				}
			}
		}
		
	}
	
	
	// HANDLERS
	
	/*
	 * getFile()
	 *
	 * Gets the remote file to $this->root_directory
	 * before the parser can run. 
	 */
	private function getFile() {
		if(!empty($this->configuration)) {
			$user_name = (string)$this->configuration->username;
			$password= (string)$this->configuration->password;
			$host = (string)$this->configuration->host;
			$port = (string)$this->configuration->port;
			$filepath = (string)$this->configuration->remote_filepath;
			$remote_file = file_get_contents("ssh2.sftp://$user_name:$password@$host:$port/$filepath");
			if(!empty($remote_file) && is_dir($this->root_directory) && is_writable($this->root_directory) && !empty($this->csv_filepath)) {
				$result = file_put_contents($this->csv_filepath, $remote_file);
				if(!empty($result)) {
					$this->log('File transfer', 'Retreived data from remote server');
					return TRUE;
				} else {
					$this->log('File transfer', 'Failed to retreive data from remote server. Bailing');
					$this->log('error', 'Failed to retreive data from remote server.');
					return FALSE;
				}
			}
		}
		return FALSE;
	}
	
	/*
	 * parseFile()
	 *
	 * Gets local file from $this->root_directory and parses
	 * it for correctness.
	 */
	private function parseFile() {
		if($this->openCSV() === TRUE) {
			if($this->rowChecker() === TRUE) {
				$this->log('Processing', "Processing " . count($this->good_rows) . " rows");
				foreach($this->good_rows as $key=>$row) {
					$upsert_result = $this->upsertCustomer($key, $row);
				}
				return TRUE;
			}
		}
		return FALSE;
	}
	
	// UTILITIES
	
	/**
	 * upsertCustomer()
	 * Looks for an existing customer record and
	 *   a. Adds a new record if none exists, or
	 *   b. Updates any existing record
	 */
	private function upsertCustomer($customer_code, $customer_details) {
		$query = 'SELECT entity_id FROM customer_entity_int WHERE attribute_id = 146 AND value = ' . $customer_details['customercode'];
		$customer_id= $this->db->fetchOne($query);
		if(!empty($customer_id) && !empty($customer_details['email'])) {
			// check the email
			$email_check = $this->existingEmailCheck($customer_id, $customer_details['email']);
			if($email_check === TRUE) {
				// update
				$customer = Mage::getModel('customer/customer')->load($customer_id);
				$customer_data = $customer->getData();
				$default_billing     = Mage::getModel('customer/address')->load($customer_data['default_billing']);
				$default_shipping    = Mage::getModel('customer/address')->load($customer_data['default_shipping']);
				if(!empty($customer_data)) {
					// set base details
					$customer_object = $this->setCustomer('existing', $customer, $customer_details);
					if(!empty($customer_data['default_billing'])) {
						// set billing address
						$billing_address_obj = $this->setAddress($customer_object, 'billing', $customer_details['billing'], $default_billing);
					}
					if(!empty($customer_data['default_shipping'])) {
						// set shipping address
						$shipping_address_obj = $this->setAddress($customer_object, 'shipping', $customer_details['shipping'], $default_shipping);
					}
				}
				$this->log('success', 'Updated existing customer [' . $customer_details['customercode'] .']');
			} else {
				$this->log('error', 'Failed to update existing customer', array('details'=>$customer_details));
				$this->bad_rows[] = $customer_details;
			}
		} elseif(!empty($customer_details['email'])) {
			// check the email
			// add
			$email_check = $this->existingEmailCheck($customer_id, $customer_details['email']);
			if($email_check === TRUE) {
				$customer_object = $this->setCustomer('new', FALSE, $customer_details);
				if(!empty($customer_object) && ($id = $customer_object->getId())) {
					if(!empty($customer_details['billing'])) {
						// set billing address
						$billing_address_obj = $this->setAddress($customer_object, 'billing', $customer_details['billing']);
					}
					if(!empty($customer_details['shipping'])) {
						// set shipping address
						$shipping_address_obj = $this->setAddress($customer_object, 'shipping', $customer_details['shipping']);
					}
					$this->log('success', 'Added new customer [' . $customer_details['customercode'] .']');					
				} else {
					$this->log('error', 'Failed to add new customer', array('details'=>$customer_details));
				}
			}
		}
	}
	
	private function setCustomer($type='new', $customer_object=FALSE, $details) {
		
		if(($type === 'new') && empty($customer_object) && !empty($details['customercode'])) {
			$website_id = Mage::app()->getWebsite()->getId();
			$store = Mage::app()->getStore();
			$customer_object = Mage::getModel("customer/customer");
			$customer_object->setWebsiteId($websiteId)
			->setStore($store)
			->setPassword('123456789');
			$customer_object->setCustomercode($details['customercode']);
        }
        
		$customer_object->setFirstname($details['firstname'])
			->setLastname ($details['lastname'])
			->setGroupId($details['group_id'])
			->setEmail($details['email'])
			->setCompany($details['company'])
			->setDeliverycode($details['deliverycode']);
		
		try {
			$customer_object->save();
		}
		catch (Exception $e) {
			$this->setError($customer_object->getCustomercode(), 'Customer save error', $e->getMessage());
			$this->log('error', 'Customer save error.', $e->getMessage());
//			Zend_Debug::dump($e->getMessage());
		}

		return $customer_object;
		
	}
	
	private function setAddress($customer_object, $type, $details, $existing=FALSE) {
		if(empty($existing)) {
			$address = Mage::getModel("customer/address");
			$address->setCustomerId($customer_object->getId())->setSaveInAddressBook('1');
			if($type === 'billing') {
				$address->setIsDefaultBilling('1');
			} elseif ($type === 'shipping') {
				$address->setIsDefaultShipping('1');
			}
		} else {
			$address = $existing;
		} 
		
		$address->setFirstname($details['firstname'])
			->setLastname($details['lastname'])
			->setCountryId($details['country_id'])
			->setRegionId($details['region'])
			->setPostcode($details['postcode'])
			->setCity($details['city'])
			->setTelephone($details['telephone'])
			->setCompany($details['company'])
			->setStreet($details['street']);
		
		try{
			$address->save();
		}
		catch (Exception $e) {
			$this->setError($customer_object->getCustomercode(), 'Address save error', $e->getMessage());
			$this->log('error', 'Address save error.', $e->getMessage());
//			Zend_Debug::dump($e->getMessage());
		}
		
		return $address;
	}
	
	/**
	 * openCSV()
	 * Attempts to open and CSV load the supplied file
	 *
	 * @return boolean
	 */
	private function openCSV() {
		$this->checkCSVExists($this->csv_basename);
		$this->all_rows= array();
		$row = 1;
		$column_count = array();
		if (($handle = fopen($this->csv_filepath, "r")) !== FALSE) {
			while (($data = fgetcsv($handle, 1000, ",")) !== FALSE) {
				$num = count($data);
				$column_count[$num][] = $row;
				$mapped_data      = $this->mapRow($data);
				// the customer code must be unique
				$cust_code_key = 'c'.$mapped_data['customercode'];
				$this->all_rows[$cust_code_key] = $mapped_data;
				$row++;
			}
			fclose($handle);
		} else {
			$this->log('Bad CSV', 'This CSV file cannot be opened by PHP', $column_count);
			$this->log('error', 'This CSV file cannot be opened by PHP', array('column_count'=>$column_count));
		}
		
		$header =  array_shift($this->all_rows);
		// there should be just one column count
		if(count($column_count) > 1) {
			$this->log('Bad CSV', 'This CSV file has a bad column count', $column_count);
			$this->log('error', 'This CSV file  has a bad column count', array('column_count'=>$column_count));
			return FALSE;
		} elseif(count($this->all_rows) < 1) {
			// there will be at least one row
			$this->log('error', 'There are no rows in this CSV file: ' . $this->csv_filepath);
			die('There are no rows in this CSV file: ' . $this->csv_filepath. PHP_EOL);
		} elseif(count($column_count) < 1) {
			$this->log('error', 'There are no columns in this CSV file: ' . $this->csv_filepath);
			die('There are no columns in this CSV file: ' . $this->csv_filepath. PHP_EOL);
		} elseif(empty($column_count[$this->row_count])) {
			$this->log('error', 'There are an unexpected number of columns in this CSV file. Expected ' . $this->row_count);
			die('There are an unexpected number of columns in this CSV file. Expected ' . $this->row_count . PHP_EOL);
		}
		return TRUE;
	}
	
	/**
	 * rowChecker()
	 * Parses all CSV rows for correctness
	 *
	 * @return boolean
	 */
	private function rowChecker() {
		$unique_emails         = array();
		$unique_customer_codes = array();
		foreach($this->all_rows as $key=>$row) {
			$unique_emails[$row['email']][] = $key;
			$this->good_rows[$key] = $row;
		}

		$this->emailChecker($unique_emails);
		
		$this->log('Processing customers', 'Initial parse', array(
				count($this->all_rows) . ' rows in total', 
				count($this->good_rows) . ' good rows',
				count($this->incorrect_rows) . ' not so good rows',
		));
		
		
		
		return TRUE;
	}
	
	/**
	 * existingEmailCheck()
	 * If an email exists and it is not that of the current customer
	 * return FALSE.
	 * 
	 * @param unknown $customer_id
	 * @param unknown $email_address
	 * @return boolean
	 */
	private function existingEmailCheck($customer_id, $email_address) {
		$customer = Mage::getModel('customer/customer');
		$customer->setWebsiteId(Mage::app()->getWebsite()->getId());
		$customer->loadByEmail($email_address);
		$id = $customer->getId();
		if(!empty($id) && ($id !== $customer_id)) {
			$this->log('error', "This email address [$email_address] is already in use");
			return FALSE;
		}
		return TRUE;
	}
	
	private function setError($customer_code, $type, $message) {
		if(empty($this->errors['c' . $customer_code])) {
			$this->errors['c' . $customer_code] = array();
		}
		$this->errors['c' . $customer_code][] = array('type'=>$type, 'message'=>$message);
		
	}
	
	private function mapRow($row) {
		$new_data = array();
		if(!empty($row) && is_array($row)) {
			$new_data['customercode']        = trim($row[0]);
			$new_data['firstname']    = trim($row[1]);
			if(empty(trim($row[2]))) {
				$this->log('error', $new_data['customercode'] . ' has no last name');
				$this->setError($new_data['customercode'], 'Data error', 'No Last Name');
			}
			$new_data['lastname']     = trim($row[2]);
			$new_data['group_id']        = trim($row[22]);
			$new_data['email']           = trim($row[23]);
			$new_data['company']         = trim($row[24]);
			$new_data['deliverycode']    = trim($row[25]);
			$new_data['Website']         = trim($row[26]);
			$new_data['billing']      = array(
					'firstname'         => trim($row[1]),
					'lastname'          => trim($row[2]),
					// not required in Magento
					//'AccContact'      => trim($row[3]),
					'telephone'        => trim($row[4]),
					'company'          => trim($row[5]),
					'street'           => trim($row[6]),
					'AccAdd02'         => trim($row[7]),
					'city'             => trim($row[8]),
					'region'           => trim($row[9]),
					'postcode'         => trim($row[10]),
					'country_id'       => 'NZ',
			);
			$new_data['shipping']    = array(
					'firstname'        => trim($row[11]),
					'lastname'         => trim($row[12]),
					// not required in Magento
					//'DelContact'      => trim($row[13]),
					'telephone'        => trim($row[14]),
					'company'          => trim($row[15]),
					'street'           => trim($row[16]),
					'DelAdd02'         => trim($row[17]),
					'city'             => trim($row[18]),
					'region'           => trim($row[19]),
					'postcode'         => trim($row[20]),
					'company'          => trim($row[24]),
					'country_id'       => 'NZ',
					// not supported in Magento
//					'delivery_instructions' => trim($row[21]),
			);
		}
		return $new_data;
	}
	
	
	// data checkers
	
	private function emailChecker($unique_emails) {
		foreach($unique_emails as $key=>$value) {
			if(count($value) > 1) {
				foreach($value as $invalid_customer_record) {
					$this->log('error', 'This record duplicates an existing email address in the CSV file', array('invalid_customer_record'=>$this->good_rows[$invalid_customer_record]));
					// add to the invalid array
					$this->incorrect_rows[$invalid_customer_record]  = $this->all_rows[$invalid_customer_record];
					// remove it from the good array
					unset($this->good_rows[$invalid_customer_record]);
				}
			}
		}
	}
	
	// file system
	private function checkRootDirectory() {
		if(!is_dir($this->root_directory)) {
			die('Please create a directory for your data here: ' . $this->root_directory. PHP_EOL);
		}
	}
	
	private function checkCSVExists($csv_basename) {
		$this->csv_filepath = $this->root_directory . DIRECTORY_SEPARATOR . $csv_basename;
		if(!is_file($this->csv_filepath)) {
			die('We were expecting there to be a CSV file here: ' . $this->csv_filepath. PHP_EOL);
		}
	}
	
	
	// logging
	private function log($type, $message, $data=array()) {
		if($this->debugging === TRUE) {
			// display it
			echo "$type\n";
			echo str_repeat('-', strlen($type)) . PHP_EOL;
			echo "$message\n";
			if(!empty($data)) {
				var_dump($data);
			}
			echo PHP_EOL;
		} 
		
		if ($type === 'error') {
			if(!empty($this->error_log) || is_writable($this->root_directory)) {
				$fh = fopen($this->error_log, 'a+');
				$string = $message;
				if(!empty($data)) {
					$string .=  ' – ' .print_r($data, 1);
				}
				$string .= PHP_EOL;
				fwrite($fh, $string);
				fclose($fh);
			}
		} elseif ($type === 'success') {
			if(!empty($this->success_log) || is_writable($this->root_directory)) {
				$fh = fopen($this->success_log, 'a+');
				$string = $message;
				if(!empty($data)) {
					$string .=  ' – ' .print_r($data, 1);
				}
				$string .= PHP_EOL;
				fwrite($fh, $string);
				fclose($fh);
			}
			
		}
		
	}
	
	public function usageHelp() {
		return <<<USAGE
Usage:  php -f mc_customer_importer.php -- [options]
				
  debug         Show Compilation State
  help          This help
				
				
USAGE;
	}
	
} // end of Mage_Shell_MCCustomerImporter class


// run it
$shell = new Mage_Shell_MCCustomerImporter();
$shell->run();